package be.cytomine.image.multidim

/*
* Copyright (c) 2009-2017. Authors: see NOTICE file.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

import be.cytomine.command.*
import be.cytomine.project.Project
import be.cytomine.security.SecUser
import be.cytomine.utils.ModelService
import be.cytomine.utils.Task
import grails.gorm.transactions.Transactional

import static org.springframework.security.acls.domain.BasePermission.READ

@Transactional
class ImageGroupService extends ModelService {

    static transactional = true

    def cytomineService
    def transactionService
    def userAnnotationService
    def algoAnnotationService
    def dataSource
    def reviewedAnnotationService
    def imageSequenceService
    def securityACLService
    def abstractImageService

    def currentDomain() {
        return ImageGroup
    }

    def read(def id) {
        def image = ImageGroup.read(id)
        if(image) {
            securityACLService.check(image.container(),READ)
        }
        image
    }

    def list(Project project) {
        securityACLService.check(project,READ)
        return ImageGroup.findAllByProject(project)
    }


    /**
     * Add the new domain with JSON data
     * @param json New domain data
     * @return Response structure (created domain data,..)
     */
    def add(def json) {
        securityACLService.check(json.project,Project,READ)
        SecUser currentUser = cytomineService.getCurrentUser()
        json.user = currentUser.id
        synchronized (this.getClass()) {
            Command c = new AddCommand(user: currentUser)
            executeCommand(c,null,json)
        }
    }

    /**
     * Update this domain with new data from json
     * @param domain Domain to update
     * @param jsonNewData New domain datas
     * @return  Response structure (new domain data, old domain data..)
     */
    def update(ImageGroup domain, def jsonNewData) {
        securityACLService.check(domain.container(),READ)
        securityACLService.check(jsonNewData.project,Project,READ)

        SecUser currentUser = cytomineService.getCurrentUser()
        Command c = new EditCommand(user: currentUser)
        executeCommand(c,domain,jsonNewData)
    }

    /**
     * Delete this domain
     * @param domain Domain to delete
     * @param transaction Transaction link with this command
     * @param task Task for this command
     * @param printMessage Flag if client will print or not confirm message
     * @return Response structure (code, old domain,..)
     */
    def delete(ImageGroup domain, Transaction transaction = null, Task task = null, boolean printMessage = true) {
        securityACLService.check(domain.container(),READ)
        SecUser currentUser = cytomineService.getCurrentUser()
        Command c = new DeleteCommand(user: currentUser,transaction:transaction)
        return executeCommand(c,domain,null)
    }

    def getStringParamsI18n(def domain) {
        return [domain.id,  domain.name, domain.project.name]
    }

    def deleteDependentImageSequence(ImageGroup group, Transaction transaction, Task task = null) {
        ImageSequence.findAllByImageGroup(group).each {
            imageSequenceService.delete(it,transaction,null,false)
        }
    }

    def characteristics(ImageGroup imageGroup){
        def poss = ImageSequence.findAllByImageGroup(imageGroup)
        def z = []
        def t = []
        def c = []
        def s = []

        poss.each {
            z << it.zStack
            t << it.time
            c << it.channel
            s << it.slice
        }

        z = z.unique().sort()
        t = t.unique().sort()
        c = c.unique().sort()
        s = s.unique().sort()
        return [slice:s,zStack:z,time:t,channel:c, imageGroup:imageGroup.id]

    }

    def thumb(Long id, int maxSize) {
        ImageGroup imageGroup = ImageGroup.get(id)
        def sequences = ImageSequence.findAllByImageGroupAndSliceAndTimeAndChannel(imageGroup,0,0,0)
        def zs = sequences.collect{it.zStack}
        int zMean = (zs.max() - zs.min())/2
        return abstractImageService.thumb(sequences.find{it.zStack == zMean}.image.baseImage.id, maxSize)
    }
}
