package be.cytomine.image.multidim

/*
* Copyright (c) 2009-2017. Authors: see NOTICE file.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

import be.cytomine.CytomineDomain
import be.cytomine.api.UrlApi
import be.cytomine.exception.AlreadyExistException
import be.cytomine.project.Project
import be.cytomine.utils.JSONUtils
import rest.api.doc.annotation.RestApiObject
import rest.api.doc.annotation.RestApiObjectField

/**
 * Created by IntelliJ IDEA.
 * User: lrollus
 * Date: 18/05/11
 * Time: 8:33
 * A group of image with diff dimension
 */
@RestApiObject(name = "Image group", description = "A group of image from the same source with different dimension")
class ImageGroup extends CytomineDomain {

    @RestApiObjectField(description = "The name of the project")
    String name

    @RestApiObjectField(description = "The image group project")
    Project project

    static constraints = {
    }

    static mapping = {
        tablePerHierarchy false
        id generator: "assigned"
        sort "id"
    }

    def beforeValidate() {
        super.beforeValidate()
        if (!name) {
            name = id
        }
    }

    /**
     * Check if this domain will cause unique constraint fail if saving on database
     */
    void checkAlreadyExist() {
        ImageGroup.withNewSession {
            ImageGroup imageAlreadyExist = ImageGroup.findByNameAndProject(name,project)
            if (imageAlreadyExist != null && (imageAlreadyExist.id != id)) {
                throw new AlreadyExistException("ImageGroup with name=" + name + " and project=" + project + "  already exists")
            }
        }
    }

    /**
     * Insert JSON data into domain in param
     * @param domain Domain that must be filled
     * @param json JSON containing data
     * @return Domain with json data filled
     */
    static ImageGroup insertDataIntoDomain(def json, def domain = new ImageGroup()) {
        domain.id = JSONUtils.getJSONAttrLong(json,'id',null)
        domain.created = JSONUtils.getJSONAttrDate(json, "created")
        domain.updated = JSONUtils.getJSONAttrDate(json, "updated")
        domain.project = JSONUtils.getJSONAttrDomain(json, "project", new Project(), true)
        domain.name = JSONUtils.getJSONAttrStr(json, "name");
        return domain;
    }

    /**
     * Define fields available for JSON response
     * @param domain Domain source for json value
     * @return Map with fields (keys) and their values
     */
    static def getDataFromDomain(def domain) {
        def returnArray = CytomineDomain.getDataFromDomain(domain)
        returnArray['name'] = domain?.name
        returnArray['project'] = domain?.project?.id
        try {
            returnArray['thumb'] = UrlApi.getThumbMultiDimImage(domain.id, 512)
        } catch (Exception e) {
            returnArray['thumb'] = 'NO THUMB:' + e.toString()
        }
        return returnArray
    }

    /**
     * Get the container domain for this domain (usefull for security)
     * @return Container of this domain
     */
    public CytomineDomain container() {
        return project.container();
    }
}
