package be.cytomine.social

import be.cytomine.exception.CytomineException
import be.cytomine.exception.ObjectNotFoundException
import be.cytomine.RestController

/*
* Copyright (c) 2009-2017. Authors: see NOTICE file.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
import be.cytomine.image.ImageInstance
import be.cytomine.security.SecUser
import be.cytomine.security.User
import rest.api.doc.annotation.RestApi
import rest.api.doc.annotation.RestApiMethod
import rest.api.doc.annotation.RestApiParam
import rest.api.doc.annotation.RestApiParams
import rest.api.doc.pojo.RestApiParamType

/**
 * Controller for user position
 * Position of the user (x,y) on an image for a time
 */
@RestApi(name = "Social| user position services", description = "Methods for managing user positions in an image")
class UserPositionController extends RestController {

    def cytomineService
    def imageInstanceService
    def secUserService
    def dataSource
    def projectService
    def mongo
    def userPositionService

    @RestApiMethod(description="Record the position of the current user on an image.")
    @RestApiParams(params=[
            @RestApiParam(name="image", type="long", paramType = RestApiParamType.QUERY, description = "The image id (Mandatory)"),
            @RestApiParam(name="topLeftX", type="double", paramType = RestApiParamType.QUERY, description = "Top Left X coordinate of the user viewport"),
            @RestApiParam(name="topRightX", type="double", paramType = RestApiParamType.QUERY, description = "Top Right X coordinate of the user viewport"),
            @RestApiParam(name="bottomLeftX", type="double", paramType = RestApiParamType.QUERY, description = "Bottom Left X coordinate of the user viewport"),
            @RestApiParam(name="bottomRightX", type="double", paramType = RestApiParamType.QUERY, description = "Bottom Right X coordinate of the user viewport"),
            @RestApiParam(name="topLeftY", type="double", paramType = RestApiParamType.QUERY, description = "Top Left Y coordinate of the user viewport"),
            @RestApiParam(name="topRightY", type="double", paramType = RestApiParamType.QUERY, description = "Top Right Y coordinate of the user viewport"),
            @RestApiParam(name="bottomLeftY", type="double", paramType = RestApiParamType.QUERY, description = "Bottom Left Y coordinate of the user viewport"),
            @RestApiParam(name="bottomRightY", type="double", paramType = RestApiParamType.QUERY, description = "Bottom Right Y coordinate of the user viewport"),
            @RestApiParam(name="zoom", type="integer", paramType = RestApiParamType.QUERY, description = "Zoom level in the user viewport")
    ])
    def add() {
        try {
            responseSuccess(userPositionService.add(request.JSON))
        } catch (CytomineException e) {
            log.error(e.toString())
            response([success: false, errors: e.msg], e.code)
        }
    }


    @RestApiMethod(description="Get the last position for a user and an image.")
    @RestApiParams(params=[
            @RestApiParam(name="id", type="long", paramType = RestApiParamType.PATH, description = "The image id (Mandatory)"),
            @RestApiParam(name="user", type="long", paramType = RestApiParamType.PATH, description = "The user id (Mandatory)")
    ])
    def lastPositionByUser() {
        ImageInstance image = imageInstanceService.read(params.id)
        SecUser user = secUserService.read(params.user)
        responseSuccess(userPositionService.lastPositionByUser(image, user))
    }

    @RestApiMethod(description="Summarize the UserPosition entries.")
    @RestApiParams(params=[
            @RestApiParam(name="image", type="long", paramType = RestApiParamType.PATH, description = "The image id"),
            @RestApiParam(name="user", type="long", paramType = RestApiParamType.QUERY, description = "The user id", required=false),
            @RestApiParam(name="afterThan", type="long", paramType = RestApiParamType.QUERY, description = "A date. Will select all the entries created after this date.", required=false),
            @RestApiParam(name="beforeThan", type="long", paramType = RestApiParamType.QUERY, description = "A date. Will select all the entries created before this date.", required=false),
            @RestApiParam(name="showDetails", type="boolean", paramType = RestApiParamType.QUERY, description = "Optional. If true, will give the complete list", required=false),
    ])
    def list() {
        ImageInstance image = imageInstanceService.read(params.image)
        User user = secUserService.read(params.user)
        if(params.user != null && user == null) throw new ObjectNotFoundException("Invalid user")

        Long afterThan = params.long("afterThan")
        Long beforeThan = params.long("beforeThan")
        if(params.getBoolean("showDetails")){
            responseSuccess(userPositionService.list(image, user, afterThan, beforeThan))
        } else {
            responseSuccess(userPositionService.summarize(image, user, afterThan, beforeThan))
        }
    }

    @RestApiMethod(description="Get users that have opened an image recently.")
    @RestApiParams(params=[
            @RestApiParam(name="id", type="long", paramType = RestApiParamType.PATH, description = "The image id (Mandatory)"),
    ])
    def listOnlineUsersByImage() {
        ImageInstance image = imageInstanceService.read(params.id)
        responseSuccess(userPositionService.listOnlineUsersByImage(image))
    }
}
