
test("TinyColor initialization", function() {
  ok (typeof tinycolor != "undefined", "tinycolor is initialized on the page");
  ok (typeof tinycolor("red") == "object", "tinycolor is able to be instantiated.");
  var r = tinycolor("red");
  ok (tinycolor(r) === r, "when given a tinycolor instance, tinycolor() returns it.");
  ok (new tinycolor(r) === r, "when given a tinycolor instance, new tinycolor() returns it.");
  equal (tinycolor("red", { format: "hex" }).toString(), "#ff0000", "tinycolor options are being parsed");
  equal (tinycolor.fromRatio({r: 1, g: 0, b: 0 }, { format: "hex" }).toString(), "#ff0000", "tinycolor options are being parsed");
});

// Taken from convertWikipediaColors.html
var conversions = [
  {"hex":"#FFFFFF","hex8":"#FFFFFFFF","rgb":{"r":"100.0%","g":"100.0%","b":"100.0%"},"hsv":{"h":"0","s":"0.000","v":"1.000"},"hsl":{"h":"0","s":"0.000","l":"1.000"}},
  {"hex":"#808080","hex8":"#FF808080","rgb":{"r":"050.0%","g":"050.0%","b":"050.0%"},"hsv":{"h":"0","s":"0.000","v":"0.500"},"hsl":{"h":"0","s":"0.000","l":"0.500"}},
  {"hex":"#000000","hex8":"#FF000000","rgb":{"r":"000.0%","g":"000.0%","b":"000.0%"},"hsv":{"h":"0","s":"0.000","v":"0.000"},"hsl":{"h":"0","s":"0.000","l":"0.000"}},
  {"hex":"#FF0000","hex8":"#FFFF0000","rgb":{"r":"100.0%","g":"000.0%","b":"000.0%"},"hsv":{"h":"0.0","s":"1.000","v":"1.000"},"hsl":{"h":"0.0","s":"1.000","l":"0.500"}},
  {"hex":"#BFBF00","hex8":"#FFBFBF00","rgb":{"r":"075.0%","g":"075.0%","b":"000.0%"},"hsv":{"h":"60.0","s":"1.000","v":"0.750"},"hsl":{"h":"60.0","s":"1.000","l":"0.375"}},
  {"hex":"#008000","hex8":"#FF008000","rgb":{"r":"000.0%","g":"050.0%","b":"000.0%"},"hsv":{"h":"120.0","s":"1.000","v":"0.500"},"hsl":{"h":"120.0","s":"1.000","l":"0.250"}},
  {"hex":"#80FFFF","hex8":"#FF80FFFF","rgb":{"r":"050.0%","g":"100.0%","b":"100.0%"},"hsv":{"h":"180.0","s":"0.500","v":"1.000"},"hsl":{"h":"180.0","s":"1.000","l":"0.750"}},
  {"hex":"#8080FF","hex8":"#FF8080FF","rgb":{"r":"050.0%","g":"050.0%","b":"100.0%"},"hsv":{"h":"240.0","s":"0.500","v":"1.000"},"hsl":{"h":"240.0","s":"1.000","l":"0.750"}},
  {"hex":"#BF40BF","hex8":"#FFBF40BF","rgb":{"r":"075.0%","g":"025.0%","b":"075.0%"},"hsv":{"h":"300.0","s":"0.667","v":"0.750"},"hsl":{"h":"300.0","s":"0.500","l":"0.500"}},
  {"hex":"#A0A424","hex8":"#FFA0A424","rgb":{"r":"062.8%","g":"064.3%","b":"014.2%"},"hsv":{"h":"61.8","s":"0.779","v":"0.643"},"hsl":{"h":"61.8","s":"0.638","l":"0.393"}},
  {"hex":"#1EAC41","hex8":"#FF1EAC41","rgb":{"r":"011.6%","g":"067.5%","b":"025.5%"},"hsv":{"h":"134.9","s":"0.828","v":"0.675"},"hsl":{"h":"134.9","s":"0.707","l":"0.396"}},
  {"hex":"#B430E5","hex8":"#FFB430E5","rgb":{"r":"070.4%","g":"018.7%","b":"089.7%"},"hsv":{"h":"283.7","s":"0.792","v":"0.897"},"hsl":{"h":"283.7","s":"0.775","l":"0.542"}},
  {"hex":"#FEF888","hex8":"#FFFEF888","rgb":{"r":"099.8%","g":"097.4%","b":"053.2%"},"hsv":{"h":"56.9","s":"0.467","v":"0.998"},"hsl":{"h":"56.9","s":"0.991","l":"0.765"}},
  {"hex":"#19CB97","hex8":"#FF19CB97","rgb":{"r":"009.9%","g":"079.5%","b":"059.1%"},"hsv":{"h":"162.4","s":"0.875","v":"0.795"},"hsl":{"h":"162.4","s":"0.779","l":"0.447"}},
  {"hex":"#362698","hex8":"#FF362698","rgb":{"r":"021.1%","g":"014.9%","b":"059.7%"},"hsv":{"h":"248.3","s":"0.750","v":"0.597"},"hsl":{"h":"248.3","s":"0.601","l":"0.373"}},
  {"hex":"#7E7EB8","hex8":"#FF7E7EB8","rgb":{"r":"049.5%","g":"049.3%","b":"072.1%"},"hsv":{"h":"240.5","s":"0.316","v":"0.721"},"hsl":{"h":"240.5","s":"0.290","l":"0.607"}}
];


module("Color translations");

test("Color Equality", function() {
  for (var i = 0; i < conversions.length; i++) {
    var c =  conversions[i];
    var tiny =  tinycolor(c.hex);

    ok (true,
      "Testing " + c.hex + ": " + tiny.toRgbString() + " " + tiny.toPercentageRgbString() + " " + tiny.toHsvString() + " " + tiny.toHslString() + " " + tiny.toHexString() +
      "Original: " + JSON.stringify(c.rgb) + " " + JSON.stringify(c.hsv) + " " + JSON.stringify(c.hsl)
    );
    ok (tinycolor.equals(c.rgb, c.hex), "RGB equals hex " + c.hex);
    ok (tinycolor.equals(c.rgb, c.hex8), "RGB equals hex " + c.hex);
    ok (tinycolor.equals(c.rgb, c.hsl), "RGB equals HSL " + c.hex);
    ok (tinycolor.equals(c.rgb, c.hsv), "RGB equals HSV " + c.hex);
    ok (tinycolor.equals(c.rgb, c.rgb), "RGB equals RGB " + c.hex);

    ok (tinycolor.equals(c.hex, c.hex), "hex equals hex " + c.hex);
    ok (tinycolor.equals(c.hex, c.hex8), "hex equals hex8 " + c.hex);
    ok (tinycolor.equals(c.hex, c.hsl), "hex equals HSL " + c.hex);
    ok (tinycolor.equals(c.hex, c.hsv), "hex equals HSV " + c.hex);

    ok (tinycolor.equals(c.hsl, c.hsv), "HSL equals HSV " + c.hex);

  }
});

module("Ratio Parsing");
test("With Ratio", function() {
  equal (tinycolor.fromRatio({r: 1, g: 1, b: 1}).toHexString(), "#ffffff", "white");
  equal (tinycolor.fromRatio({r: 1, g: 0, b: 0, a: .5 }).toRgbString(), "rgba(255, 0, 0, 0.5)", "alpha works when ratio is parsed");
  equal (tinycolor.fromRatio({r: 1, g: 0, b: 0, a: 1 }).toRgbString(), "rgb(255, 0, 0)", "alpha = 1 works when ratio is parsed");
  equal (tinycolor.fromRatio({r: 1, g: 0, b: 0, a: 10 }).toRgbString(), "rgb(255, 0, 0)", "alpha > 1 works when ratio is parsed");
  equal (tinycolor.fromRatio({r: 1, g: 0, b: 0, a: -1 }).toRgbString(), "rgb(255, 0, 0)", "alpha < 1 works when ratio is parsed");
});

test("Without Ratio", function() {
  equal (tinycolor({r: 1, g: 1, b: 1}).toHexString(), "#010101", "010101");
  equal (tinycolor({r: .1, g: .1, b: .1}).toHexString(), "#000000", "000000");
  equal (tinycolor("rgb .1 .1 .1").toHexString(), "#000000", "000000");

});

module("String Parsing");

test("RGB Text Parsing", function() {

  equal (tinycolor("rgb 255 0 0").toHexString(), "#ff0000", "spaced input");
  equal (tinycolor("rgb(255, 0, 0)").toHexString(), "#ff0000", "parenthesized input");
  equal (tinycolor("rgb (255, 0, 0)").toHexString(), "#ff0000", "parenthesized spaced input");
  equal (tinycolor({ r: 255, g: 0, b: 0 }).toHexString(), "#ff0000", "object input");
  deepEqual (tinycolor({ r: 255, g: 0, b: 0 }).toRgb(), { r: 255, g: 0, b: 0, a: 1 }, "object input and compare");


  ok (tinycolor.equals({r:200, g: 100, b: 0 }, "rgb(200, 100, 0)"));
  ok (tinycolor.equals({r:200, g: 100, b: 0 }, "rgb 200 100 0"));
  ok (tinycolor.equals({r:200, g: 100, b: 0 }, "rgb 200 100 0"));
  ok (tinycolor.equals({r:200, g: 100, b: 0, a: .4 }, "rgba 200 100 0 .4"));
  ok (!tinycolor.equals({r:199, g: 100, b: 0 }, "rgba 200 100 0 1"));

  ok (!tinycolor.equals({r:199, g: 100, b: 0 }, "rgb(200, 100, 0)"));
  ok (!tinycolor.equals({r:199, g: 100, b: 0 }, "rgb 200 100 0"));
  ok (!tinycolor.equals({r:199, g: 100, b: 0 }, "rgb 200 100 0"));


  ok (tinycolor.equals(tinycolor({r:200, g: 100, b: 0 }), "rgb(200, 100, 0)"));
  ok (tinycolor.equals(tinycolor({r:200, g: 100, b: 0 }), "rgb 200 100 0"));
  ok (tinycolor.equals(tinycolor({r:200, g: 100, b: 0 }), "rgb 200 100 0"));

});

test("Percentage RGB Text Parsing", function() {

  equal (tinycolor("rgb 100% 0% 0%").toHexString(), "#ff0000", "spaced input");
  equal (tinycolor("rgb(100%, 0%, 0%)").toHexString(), "#ff0000", "parenthesized input");
  equal (tinycolor("rgb (100%, 0%, 0%)").toHexString(), "#ff0000", "parenthesized spaced input");
  equal (tinycolor({ r: "100%", g: "0%", b: "0%" }).toHexString(), "#ff0000", "object input");
  deepEqual (tinycolor({ r: "100%", g: "0%", b: "0%" }).toRgb(), { r: 255, g: 0, b: 0, a: 1 }, "object input and compare");


  ok (tinycolor.equals({r:"90%", g: "45%", b: "0%" }, "rgb(90%, 45%, 0%)"));
  ok (tinycolor.equals({r:"90%", g: "45%", b: "0%" }, "rgb 90% 45% 0%"));
  ok (tinycolor.equals({r:"90%", g: "45%", b: "0%" }, "rgb 90% 45% 0%"));
  ok (tinycolor.equals({r:"90%", g: "45%", b: "0%", a: .4 }, "rgba 90% 45% 0% .4"));
  ok (!tinycolor.equals({r:"89%", g: "45%", b: "0%" }, "rgba 90% 45% 0% 1"));

  ok (!tinycolor.equals({r:"89%", g: "45%", b: "0%" }, "rgb(90%, 45%, 0%)"));
  ok (!tinycolor.equals({r:"89%", g: "45%", b: "0%" }, "rgb 90% 45% 0%"));
  ok (!tinycolor.equals({r:"89%", g: "45%", b: "0%" }, "rgb 90% 45% 0%"));


  ok (tinycolor.equals(tinycolor({r:"90%", g: "45%", b: "0%" }), "rgb(90%, 45%, 0%)"));
  ok (tinycolor.equals(tinycolor({r:"90%", g: "45%", b: "0%" }), "rgb 90% 45% 0%"));
  ok (tinycolor.equals(tinycolor({r:"90%", g: "45%", b: "0%" }), "rgb 90% 45% 0%"));

});

test("HSL parsing", function() {
  equal (tinycolor({ h: 251, s: 100, l: .38 }).toHexString(), "#2400c2", "to hex");
  equal (tinycolor({ h: 251, s: 100, l: .38 }).toRgbString(), "rgb(36, 0, 194)", "to rgb");
  equal (tinycolor({ h: 251, s: 100, l: .38 }).toHslString(), "hsl(251, 100%, 38%)", "to hsl");
  equal (tinycolor("hsl(251, 100, 38)").toHexString(), "#2400c2", "to hex");
  equal (tinycolor("hsl(251, 100%, 38%)").toRgbString(), "rgb(36, 0, 194)", "to rgb");
  equal (tinycolor("hsl(251, 100%, 38%)").toHslString(), "hsl(251, 100%, 38%)", "to hsl");
  equal (tinycolor("hsl 100 20 10").toHslString(), "hsl(100, 20%, 10%)", "problematic hsl");
});


test("Hex Parsing", function() {

  equal (tinycolor("rgb 255 0 0").toHexString(), "#ff0000");
  equal (tinycolor("rgb 255 0 0").toHexString(true), "#f00");
  equal (tinycolor("rgba 255 0 0 0.5").toHex8String(), "#80ff0000");


  equal (tinycolor("rgb 255 0 0").toHex(), "ff0000");
  equal (tinycolor("rgb 255 0 0").toHex(true), "f00");
  equal (tinycolor("rgba 255 0 0 0.5").toHex8(), "80ff0000");

});

test("HSV Parsing", function() {

  equal (tinycolor("hsv 251.1 0.887 .918").toHsvString(), "hsv(251, 89%, 92%)");
  equal (tinycolor("hsv 251.1 0.887 0.918").toHsvString(), "hsv(251, 89%, 92%)");

});

test("Invalid Parsing", function() {

  equal (tinycolor("this is not a color").toHexString(), "#000000");

});

test("Named colors", function() {
  equal (tinycolor("aliceblue").toHex(), "f0f8ff");
  equal (tinycolor("antiquewhite").toHex(), "faebd7");
  equal (tinycolor("aqua").toHex(), "00ffff");
  equal (tinycolor("aquamarine").toHex(), "7fffd4");
  equal (tinycolor("azure").toHex(), "f0ffff");
  equal (tinycolor("beige").toHex(), "f5f5dc");
  equal (tinycolor("bisque").toHex(), "ffe4c4");
  equal (tinycolor("black").toHex(), "000000");
  equal (tinycolor("blanchedalmond").toHex(), "ffebcd");
  equal (tinycolor("blue").toHex(), "0000ff");
  equal (tinycolor("blueviolet").toHex(), "8a2be2");
  equal (tinycolor("brown").toHex(), "a52a2a");
  equal (tinycolor("burlywood").toHex(), "deb887");
  equal (tinycolor("cadetblue").toHex(), "5f9ea0");
  equal (tinycolor("chartreuse").toHex(), "7fff00");
  equal (tinycolor("chocolate").toHex(), "d2691e");
  equal (tinycolor("coral").toHex(), "ff7f50");
  equal (tinycolor("cornflowerblue").toHex(), "6495ed");
  equal (tinycolor("cornsilk").toHex(), "fff8dc");
  equal (tinycolor("crimson").toHex(), "dc143c");
  equal (tinycolor("cyan").toHex(), "00ffff");
  equal (tinycolor("darkblue").toHex(), "00008b");
  equal (tinycolor("darkcyan").toHex(), "008b8b");
  equal (tinycolor("darkgoldenrod").toHex(), "b8860b");
  equal (tinycolor("darkgray").toHex(), "a9a9a9");
  equal (tinycolor("darkgreen").toHex(), "006400");
  equal (tinycolor("darkkhaki").toHex(), "bdb76b");
  equal (tinycolor("darkmagenta").toHex(), "8b008b");
  equal (tinycolor("darkolivegreen").toHex(), "556b2f");
  equal (tinycolor("darkorange").toHex(), "ff8c00");
  equal (tinycolor("darkorchid").toHex(), "9932cc");
  equal (tinycolor("darkred").toHex(), "8b0000");
  equal (tinycolor("darksalmon").toHex(), "e9967a");
  equal (tinycolor("darkseagreen").toHex(), "8fbc8f");
  equal (tinycolor("darkslateblue").toHex(), "483d8b");
  equal (tinycolor("darkslategray").toHex(), "2f4f4f");
  equal (tinycolor("darkturquoise").toHex(), "00ced1");
  equal (tinycolor("darkviolet").toHex(), "9400d3");
  equal (tinycolor("deeppink").toHex(), "ff1493");
  equal (tinycolor("deepskyblue").toHex(), "00bfff");
  equal (tinycolor("dimgray").toHex(), "696969");
  equal (tinycolor("dodgerblue").toHex(), "1e90ff");
  equal (tinycolor("firebrick").toHex(), "b22222");
  equal (tinycolor("floralwhite").toHex(), "fffaf0");
  equal (tinycolor("forestgreen").toHex(), "228b22");
  equal (tinycolor("fuchsia").toHex(), "ff00ff");
  equal (tinycolor("gainsboro").toHex(), "dcdcdc");
  equal (tinycolor("ghostwhite").toHex(), "f8f8ff");
  equal (tinycolor("gold").toHex(), "ffd700");
  equal (tinycolor("goldenrod").toHex(), "daa520");
  equal (tinycolor("gray").toHex(), "808080");
  equal (tinycolor("grey").toHex(), "808080");
  equal (tinycolor("green").toHex(), "008000");
  equal (tinycolor("greenyellow").toHex(), "adff2f");
  equal (tinycolor("honeydew").toHex(), "f0fff0");
  equal (tinycolor("hotpink").toHex(), "ff69b4");
  equal (tinycolor("indianred ").toHex(), "cd5c5c");
  equal (tinycolor("indigo ").toHex(), "4b0082");
  equal (tinycolor("ivory").toHex(), "fffff0");
  equal (tinycolor("khaki").toHex(), "f0e68c");
  equal (tinycolor("lavender").toHex(), "e6e6fa");
  equal (tinycolor("lavenderblush").toHex(), "fff0f5");
  equal (tinycolor("lawngreen").toHex(), "7cfc00");
  equal (tinycolor("lemonchiffon").toHex(), "fffacd");
  equal (tinycolor("lightblue").toHex(), "add8e6");
  equal (tinycolor("lightcoral").toHex(), "f08080");
  equal (tinycolor("lightcyan").toHex(), "e0ffff");
  equal (tinycolor("lightgoldenrodyellow").toHex(), "fafad2");
  equal (tinycolor("lightgrey").toHex(), "d3d3d3");
  equal (tinycolor("lightgreen").toHex(), "90ee90");
  equal (tinycolor("lightpink").toHex(), "ffb6c1");
  equal (tinycolor("lightsalmon").toHex(), "ffa07a");
  equal (tinycolor("lightseagreen").toHex(), "20b2aa");
  equal (tinycolor("lightskyblue").toHex(), "87cefa");
  equal (tinycolor("lightslategray").toHex(), "778899");
  equal (tinycolor("lightsteelblue").toHex(), "b0c4de");
  equal (tinycolor("lightyellow").toHex(), "ffffe0");
  equal (tinycolor("lime").toHex(), "00ff00");
  equal (tinycolor("limegreen").toHex(), "32cd32");
  equal (tinycolor("linen").toHex(), "faf0e6");
  equal (tinycolor("magenta").toHex(), "ff00ff");
  equal (tinycolor("maroon").toHex(), "800000");
  equal (tinycolor("mediumaquamarine").toHex(), "66cdaa");
  equal (tinycolor("mediumblue").toHex(), "0000cd");
  equal (tinycolor("mediumorchid").toHex(), "ba55d3");
  equal (tinycolor("mediumpurple").toHex(), "9370db");
  equal (tinycolor("mediumseagreen").toHex(), "3cb371");
  equal (tinycolor("mediumslateblue").toHex(), "7b68ee");
  equal (tinycolor("mediumspringgreen").toHex(), "00fa9a");
  equal (tinycolor("mediumturquoise").toHex(), "48d1cc");
  equal (tinycolor("mediumvioletred").toHex(), "c71585");
  equal (tinycolor("midnightblue").toHex(), "191970");
  equal (tinycolor("mintcream").toHex(), "f5fffa");
  equal (tinycolor("mistyrose").toHex(), "ffe4e1");
  equal (tinycolor("moccasin").toHex(), "ffe4b5");
  equal (tinycolor("navajowhite").toHex(), "ffdead");
  equal (tinycolor("navy").toHex(), "000080");
  equal (tinycolor("oldlace").toHex(), "fdf5e6");
  equal (tinycolor("olive").toHex(), "808000");
  equal (tinycolor("olivedrab").toHex(), "6b8e23");
  equal (tinycolor("orange").toHex(), "ffa500");
  equal (tinycolor("orangered").toHex(), "ff4500");
  equal (tinycolor("orchid").toHex(), "da70d6");
  equal (tinycolor("palegoldenrod").toHex(), "eee8aa");
  equal (tinycolor("palegreen").toHex(), "98fb98");
  equal (tinycolor("paleturquoise").toHex(), "afeeee");
  equal (tinycolor("palevioletred").toHex(), "db7093");
  equal (tinycolor("papayawhip").toHex(), "ffefd5");
  equal (tinycolor("peachpuff").toHex(), "ffdab9");
  equal (tinycolor("peru").toHex(), "cd853f");
  equal (tinycolor("pink").toHex(), "ffc0cb");
  equal (tinycolor("plum").toHex(), "dda0dd");
  equal (tinycolor("powderblue").toHex(), "b0e0e6");
  equal (tinycolor("purple").toHex(), "800080");
  equal (tinycolor("red").toHex(), "ff0000");
  equal (tinycolor("rosybrown").toHex(), "bc8f8f");
  equal (tinycolor("royalblue").toHex(), "4169e1");
  equal (tinycolor("saddlebrown").toHex(), "8b4513");
  equal (tinycolor("salmon").toHex(), "fa8072");
  equal (tinycolor("sandybrown").toHex(), "f4a460");
  equal (tinycolor("seagreen").toHex(), "2e8b57");
  equal (tinycolor("seashell").toHex(), "fff5ee");
  equal (tinycolor("sienna").toHex(), "a0522d");
  equal (tinycolor("silver").toHex(), "c0c0c0");
  equal (tinycolor("skyblue").toHex(), "87ceeb");
  equal (tinycolor("slateblue").toHex(), "6a5acd");
  equal (tinycolor("slategray").toHex(), "708090");
  equal (tinycolor("snow").toHex(), "fffafa");
  equal (tinycolor("springgreen").toHex(), "00ff7f");
  equal (tinycolor("steelblue").toHex(), "4682b4");
  equal (tinycolor("tan").toHex(), "d2b48c");
  equal (tinycolor("teal").toHex(), "008080");
  equal (tinycolor("thistle").toHex(), "d8bfd8");
  equal (tinycolor("tomato").toHex(), "ff6347");
  equal (tinycolor("turquoise").toHex(), "40e0d0");
  equal (tinycolor("violet").toHex(), "ee82ee");
  equal (tinycolor("wheat").toHex(), "f5deb3");
  equal (tinycolor("white").toHex(), "ffffff");
  equal (tinycolor("whitesmoke").toHex(), "f5f5f5");
  equal (tinycolor("yellow").toHex(), "ffff00");
  equal (tinycolor("yellowgreen").toHex(), "9acd32");

  equal (tinycolor("#f00").toName(), "red");
  equal (tinycolor("#fa0a0a").toName(), false);
});

module("Alpha handling");
test("Invalid alpha should normalize to 1", function() {
  equal (tinycolor({r:255,g:20,b:10,a: -1}).toRgbString(), "rgb(255, 20, 10)", "Negative value");
  equal (tinycolor({r:255,g:20,b:10,a: -0}).toRgbString(), "rgba(255, 20, 10, 0)", "Negative 0");
  equal (tinycolor({r:255,g:20,b:10,a: 0}).toRgbString(), "rgba(255, 20, 10, 0)", "0");
  equal (tinycolor({r:255,g:20,b:10,a: .5}).toRgbString(), "rgba(255, 20, 10, 0.5)", ".5");
  equal (tinycolor({r:255,g:20,b:10,a: 1}).toRgbString(), "rgb(255, 20, 10)", "1");
  equal (tinycolor({r:255,g:20,b:10,a: 100}).toRgbString(), "rgb(255, 20, 10)", "Greater than 1");
  equal (tinycolor({r:255,g:20,b:10,a: "asdfasd"}).toRgbString(), "rgb(255, 20, 10)",  "Non Numeric");

  equal (tinycolor("#fff").toRgbString(), "rgb(255, 255, 255)",  "Hex should be 1");
  equal (tinycolor("rgba 255 0 0 100").toRgbString(), "rgb(255, 0, 0)",  "Greater than 1 in string parsing");
});

test("toString() with alpha set", function() {
  var redNamed = tinycolor.fromRatio({ r: 255, g: 0, b: 0, a: .5}, {format: "name"});
  var transparentNamed = tinycolor.fromRatio({ r: 255, g: 0, b: 0, a: 0 }, {format: "name"});
  var redHex = tinycolor.fromRatio({ r: 255, g: 0, b: 0, a: .5}, {format: "hex"});

  equal (redNamed.toString(), "rgba(255, 0, 0, 0.5)", "Names should default to rgba if alpha is < 1");
  equal (redHex.toString(), "rgba(255, 0, 0, 0.5)", "Hex should default to rgba if alpha is < 1");

  equal (redNamed.toString("hex"), "#ff0000", "Names should not be returned as rgba if format is specified");
  equal (redNamed.toString("hex6"), "#ff0000", "Names should not be returned as rgba if format is specified");
  equal (redNamed.toString("hex3"), "#f00", "Names should not be returned as rgba if format is specified");
  equal (redNamed.toString("name"), "#ff0000", "Semi transparent names should return hex in toString() if name format is specified");

  equal (redNamed.toName(), false, "Semi transparent names should be false in toName()");

  equal (redHex.toString(), "rgba(255, 0, 0, 0.5)", "Hex should default to rgba if alpha is < 1");
  equal (transparentNamed.toString(), "transparent", "Named color should equal transparent if alpha == 0");
});

test("setting alpha", function() {

  var hexSetter = tinycolor("rgba(255, 0, 0, 1)");
  equal (hexSetter.getAlpha(), 1, "Alpha should start as 1");
  hexSetter.setAlpha(.9);
  equal (hexSetter.getAlpha(), .9, "setAlpha should change alpha value");
  hexSetter.setAlpha(.5);
  equal (hexSetter.getAlpha(), .5, "setAlpha should change alpha value");
  hexSetter.setAlpha(0);
  equal (hexSetter.getAlpha(), 0, "setAlpha should change alpha value");
  hexSetter.setAlpha(-1);
  equal (hexSetter.getAlpha(), 1, "setAlpha with value < 0 should be bound to 1");
  hexSetter.setAlpha(2);
  equal (hexSetter.getAlpha(), 1, "setAlpha with value > 1 should be bound to 1");
  hexSetter.setAlpha();
  equal (hexSetter.getAlpha(), 1, "setAlpha with invalid value should be bound to 1");
  hexSetter.setAlpha(null);
  equal (hexSetter.getAlpha(), 1, "setAlpha with invalid value should be bound to 1");
  hexSetter.setAlpha("test");
  equal (hexSetter.getAlpha(), 1, "setAlpha with invalid value should be bound to 1");

});

test("Alpha = 0 should act differently on toName()", function() {
  equal (tinycolor({r:255,g:20,b:10,a: 0}).toName(), "transparent", "0");
  equal (tinycolor("transparent").toString(), "transparent", "toString when passed");
  equal (tinycolor("transparent").toHex(), "000000", "toHex");
});

module("Initialization from tinycolor output");
test("HSL Object", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      equal (tiny.toHexString(), tinycolor(tiny.toHsl()).toHexString(), "HSL Object");
    }
});

test("HSL String", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      var input = tiny.toRgb();
      var output = tinycolor(tiny.toHslString()).toRgb();
      var maxDiff = 2;

      equal (Math.abs(input.r - output.r) <= maxDiff, true, "toHslString red value difference <= " + maxDiff);
      equal (Math.abs(input.g - output.g) <= maxDiff, true, "toHslString green value difference <= " + maxDiff);
      equal (Math.abs(input.b - output.b) <= maxDiff, true, "toHslString blue value difference <= " + maxDiff);
    }
});
test("HSV String", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      var input = tiny.toRgb();
      var output = tinycolor(tiny.toHsvString()).toRgb();
      var maxDiff = 2;

      equal (Math.abs(input.r - output.r) <= maxDiff, true, "toHsvString red value difference <= " + maxDiff);
      equal (Math.abs(input.g - output.g) <= maxDiff, true, "toHsvString green value difference <= " + maxDiff);
      equal (Math.abs(input.b - output.b) <= maxDiff, true, "toHsvString blue value difference <= " + maxDiff);
    }
});

test("HSV Object", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      equal (tiny.toHexString(), tinycolor(tiny.toHsv()).toHexString(), "HSV Object");
    }
});
test("RGB Object", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      equal (tiny.toHexString(), tinycolor(tiny.toRgb()).toHexString(), "RGB Object");
    }
});
test("RGB String", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      equal (tiny.toHexString(), tinycolor(tiny.toRgbString()).toHexString(), "RGB String");
    }
});
test("PRGB Object", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      var input = tiny.toRgb();
      var output = tinycolor(tiny.toPercentageRgb()).toRgb();
      var maxDiff = 2;

      equal (Math.abs(input.r - output.r) <= maxDiff, true, "Red value difference <= " + maxDiff);
      equal (Math.abs(input.g - output.g) <= maxDiff, true, "Green value difference <= " + maxDiff);
      equal (Math.abs(input.b - output.b) <= maxDiff, true, "Blue value difference <= " + maxDiff);
    }
});
test("PRGB String", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      var input = tiny.toRgb();
      var output = tinycolor(tiny.toPercentageRgbString()).toRgb();
      var maxDiff = 2;

      equal (Math.abs(input.r - output.r) <= maxDiff, true, "Red value difference <= " + maxDiff);
      equal (Math.abs(input.g - output.g) <= maxDiff, true, "Green value difference <= " + maxDiff);
      equal (Math.abs(input.b - output.b) <= maxDiff, true, "Blue value difference <= " + maxDiff);
    }
});
test("Object", function() {
    for (var i = 0; i < conversions.length; i++) {
      var c =  conversions[i];
      var tiny =  tinycolor(c.hex);
      equal (tiny.toHexString(), tinycolor(tiny).toHexString(), "Object");
    }
});


module("Utilities");

test("Color equality", function() {
  ok (tinycolor.equals("#ff0000", "#ff0000"), "Same hex");
  ok (tinycolor.equals("#ff0000", "rgb(255, 0, 0)"), "Same alphas");
  ok (!tinycolor.equals("#ff0000", "rgba(255, 0, 0, .1)"), "Different alphas");
  ok (tinycolor.equals("ff0000", "#ff0000"), "Same hex");
  ok (tinycolor.equals("#f00", "#ff0000"), "Same hex");
  ok (tinycolor.equals("f00", "#ff0000"), "Same hex");
  equal (tinycolor("010101").toHexString(), "#010101");
  ok (!tinycolor.equals("#ff0000", "#00ff00"), "Different hex");
  ok (tinycolor.equals("#ff8000", "rgb(100%, 50%, 0%)"), "Percentage bounds checking");
});

test("Readability", function () {
    ok (tinycolor.readable("#000000", "#ffffff"), "white/black is readable");
    ok (!tinycolor.readable("#FF0088", "#8822AA"), "pink on pink is not readable");
    equal (tinycolor.mostReadable("#000", ["#111", "#222"]).toHexString(), "#222222", "pick most readable color");
    equal (tinycolor.mostReadable("#f00", ["#d00", "#0d0"]).toHexString(), "#00dd00", "pick most readable color");
});

test("Filters", function () {

  equal (tinycolor("red").toFilter(), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#ffff0000,endColorstr=#ffff0000)");
  equal (tinycolor("red").toFilter("blue"), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#ffff0000,endColorstr=#ff0000ff)");

  equal (tinycolor("transparent").toFilter(), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#00000000,endColorstr=#00000000)");
  equal (tinycolor("transparent").toFilter("red"), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#00000000,endColorstr=#ffff0000)");

  equal (tinycolor("#ddf0f0f0").toFilter(), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#ddf0f0f0,endColorstr=#ddf0f0f0)");
  equal (tinycolor("rgba(0, 0, 255, .5").toFilter(), "progid:DXImageTransform.Microsoft.gradient(startColorstr=#800000ff,endColorstr=#800000ff)");
});

/* Originally generated with:
var results = [];
for (var i = 0; i <= 100; i++) results.push( tinycolor.saturate("red", i).toHex() )
console.log(JSON.stringify(results))
*/
var DESATURATIONS = ["ff0000","fe0101","fc0303","fb0404","fa0505","f90606","f70808","f60909","f50a0a","f40b0b","f20d0d","f10e0e","f00f0f","ee1111","ed1212","ec1313","eb1414","e91616","e81717","e71818","e61919","e41b1b","e31c1c","e21d1d","e01f1f","df2020","de2121","dd2222","db2424","da2525","d92626","d72828","d62929","d52a2a","d42b2b","d22d2d","d12e2e","d02f2f","cf3030","cd3232","cc3333","cb3434","c93636","c83737","c73838","c63939","c43b3b","c33c3c","c23d3d","c13e3e","bf4040","be4141","bd4242","bb4444","ba4545","b94646","b84747","b64949","b54a4a","b44b4b","b34d4d","b14e4e","b04f4f","af5050","ad5252","ac5353","ab5454","aa5555","a85757","a75858","a65959","a45b5b","a35c5c","a25d5d","a15e5e","9f6060","9e6161","9d6262","9c6363","9a6565","996666","986767","966969","956a6a","946b6b","936c6c","916e6e","906f6f","8f7070","8e7171","8c7373","8b7474","8a7575","887777","877878","867979","857a7a","837c7c","827d7d","817e7e","808080"];
var SATURATIONS = ["ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000","ff0000"];
var LIGHTENS = ["ff0000","ff0505","ff0a0a","ff0f0f","ff1414","ff1a1a","ff1f1f","ff2424","ff2929","ff2e2e","ff3333","ff3838","ff3d3d","ff4242","ff4747","ff4d4d","ff5252","ff5757","ff5c5c","ff6161","ff6666","ff6b6b","ff7070","ff7575","ff7a7a","ff8080","ff8585","ff8a8a","ff8f8f","ff9494","ff9999","ff9e9e","ffa3a3","ffa8a8","ffadad","ffb3b3","ffb8b8","ffbdbd","ffc2c2","ffc7c7","ffcccc","ffd1d1","ffd6d6","ffdbdb","ffe0e0","ffe5e5","ffebeb","fff0f0","fff5f5","fffafa","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff","ffffff"];
var DARKENS = ["ff0000","fa0000","f50000","f00000","eb0000","e60000","e00000","db0000","d60000","d10000","cc0000","c70000","c20000","bd0000","b80000","b30000","ad0000","a80000","a30000","9e0000","990000","940000","8f0000","8a0000","850000","800000","7a0000","750000","700000","6b0000","660000","610000","5c0000","570000","520000","4d0000","470000","420000","3d0000","380000","330000","2e0000","290000","240000","1f0000","190000","140000","0f0000","0a0000","050000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000","000000"];

test("Combinations", function () {
  for (var i = 0; i <= 100; i++) {
    equal (tinycolor.desaturate("red", i).toHex(), DESATURATIONS[i], "Desaturation " + i + " works");
  }
  for (var i = 0; i <= 100; i++) {
    equal (tinycolor.saturate("red", i).toHex(), SATURATIONS[i], "Saturation " + i + " works");
  }
  for (var i = 0; i <= 100; i++) {
    equal (tinycolor.lighten("red", i).toHex(), LIGHTENS[i], "Lighten " + i + " works");
  }
  for (var i = 0; i <= 100; i++) {
    equal (tinycolor.darken("red", i).toHex(), DARKENS[i], "Darken " + i + " works");
  }


  equal (tinycolor.greyscale("red").toHex(), "808080", "Greyscale works")
  equal (tinycolor.complement("red").toHex(), "00ffff", "Complement works");
});


/* Too slow: 1677731 possibilities
asyncTest("Ajax load", function() {

  $.get("allhex.txt", function(d) {
    var hex = d.split('\n');
    for (var i = 0, l = hex.length; i < l; i++) {
      ok (tinycolor(hex[i]).toHex(), hex[i]);
    }
    console.log(hex.length);
      start();
  });
});
*/
